<?php
declare(strict_types=1);

error_reporting(E_ALL);
ini_set('display_errors', '1');

if (!function_exists('str_contains')) {
    function str_contains(string $haystack, string $needle): bool
    {
        return $needle === '' || strpos($haystack, $needle) !== false;
    }
}

if (!function_exists('str_starts_with')) {
    function str_starts_with(string $haystack, string $needle): bool
    {
        if ($needle === '') {
            return true;
        }
        return strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

$rootDir    = dirname(__DIR__, 1);
$configPath = $rootDir . '/config.php';
$docsDir    = $rootDir . '/docs';
$seedFile   = __DIR__ . '/mysql_seed.sql';

$state = [
    'success' => null,
    'errors'  => [],
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = [
        'host'     => trim((string)($_POST['db_host'] ?? 'localhost')),
        'port'     => trim((string)($_POST['db_port'] ?? '3306')),
        'name'     => trim((string)($_POST['db_name'] ?? '')),
        'user'     => trim((string)($_POST['db_user'] ?? '')),
        'password' => (string)($_POST['db_pass'] ?? ''),
        'charset'  => trim((string)($_POST['db_charset'] ?? 'utf8mb4')),
    ];

    foreach (['host','port','name','user'] as $field) {
        if ($input[$field] === '') {
            $state['errors'][] = "El campo {$field} es obligatorio.";
        }
    }

    if (!ctype_digit($input['port'])) {
        $state['errors'][] = 'El puerto debe ser numérico.';
    }

    if (!preg_match('/^[a-zA-Z0-9_]+$/', $input['name'])) {
        $state['errors'][] = 'El nombre de la base solo puede contener letras, números o guion bajo.';
    }

    if (!file_exists($seedFile)) {
        $state['errors'][] = 'No se encontró el archivo mysql_seed.sql';
    }

    if (!$state['errors']) {
        try {
            $dsn = sprintf(
                'mysql:host=%s;port=%s;dbname=%s;charset=%s',
                $input['host'],
                $input['port'],
                $input['name'],
                $input['charset']
            );

            $pdo = new PDO($dsn, $input['user'], $input['password'], [
                PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES   => false,
            ]);

            runSchema($pdo, $seedFile);
            persistConfig($configPath, $input);

            if (!is_dir($docsDir)) {
                if (!mkdir($docsDir, 0775, true) && !is_dir($docsDir)) {
                    throw new RuntimeException("No se pudo crear el directorio de documentación: {$docsDir}");
                }
            }

            $schema = fetchSchema($pdo);
            $basePath = detectBasePath();

            $openapi = buildOpenApiDocument($schema, $basePath);
            $jsonOptions = JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES;
            file_put_contents($docsDir . '/openapi.json', json_encode($openapi, $jsonOptions));
            file_put_contents($docsDir . '/swagger.json', json_encode($openapi, $jsonOptions));

            $state['success'] = [
                'message' => 'Instalación completada correctamente.',
                'links'   => [
                    'API'     => $basePath,
                    'Docs'    => rtrim($basePath, '/') . '/docs/openapi.json',
                    'Swagger' => rtrim($basePath, '/') . '/docs/swagger.json',
                ],
            ];
        } catch (Throwable $e) {
            $state['errors'][] = 'Error durante la instalación: ' . $e->getMessage();
        }
    }
}

function runSchema(PDO $pdo, string $filePath): void
{
    $contents = file_get_contents($filePath);
    if ($contents === false) {
        throw new RuntimeException('No se pudo leer mysql_seed.sql');
    }

    $contents = preg_replace('/\/\*.+?\*\//s', '', $contents);
    $contents = preg_replace('/^\s*--.*$/m', '', $contents);
    $contents = preg_replace('/^\s*#.*/m', '', $contents);

    foreach (splitSqlStatements($contents) as $statement) {
        $trimmed = trim($statement);
        if ($trimmed === '') {
            continue;
        }
        $pdo->exec($trimmed);
    }
}

function splitSqlStatements(string $sql): array
{
    $statements = [];
    $buffer     = '';
    $inString   = false;
    $stringChar = '';

    $len = strlen($sql);
    for ($i = 0; $i < $len; $i++) {
        $char = $sql[$i];
        $buffer .= $char;

        if ($inString) {
            if ($char === $stringChar && ($i === 0 || $sql[$i - 1] !== '\\')) {
                $inString = false;
            }
            continue;
        }

        if ($char === '\'' || $char === '"') {
            $inString   = true;
            $stringChar = $char;
            continue;
        }

        if ($char === ';') {
            $statements[] = $buffer;
            $buffer       = '';
        }
    }

    if (trim($buffer) !== '') {
        $statements[] = $buffer;
    }

    return $statements;
}

function persistConfig(string $path, array $input): void
{
    $content = "<?php\nreturn " . var_export([
        'db' => [
            'host'    => $input['host'],
            'port'    => $input['port'],
            'name'    => $input['name'],
            'user'    => $input['user'],
            'pass'    => $input['password'],
            'charset' => $input['charset'],
        ],
    ], true) . ";\n";

    if (file_put_contents($path, $content) === false) {
        throw new RuntimeException('No se pudo escribir el archivo de configuración.');
    }
}

function fetchSchema(PDO $pdo): array
{
    $tablesData = [];
    $tables     = $pdo->query('SHOW TABLES')->fetchAll(PDO::FETCH_COLUMN);

    foreach ($tables as $table) {
        $columns = $pdo->query("SHOW FULL COLUMNS FROM `$table`")->fetchAll(PDO::FETCH_ASSOC);
        $primary = null;
        $keys    = $pdo->query("SHOW KEYS FROM `$table` WHERE Key_name = 'PRIMARY'")->fetchAll(PDO::FETCH_ASSOC);

        if ($keys) {
            $primary = $keys[0]['Column_name'] ?? null;
        }

        $tablesData[$table] = [
            'columns'    => $columns,
            'primaryKey' => $primary,
        ];
    }

    return $tablesData;
}

function detectBasePath(): string
{
    $script = $_SERVER['SCRIPT_NAME'] ?? '/api/install/index.php';
    $apiBase = dirname(dirname($script));
    return rtrim($apiBase, '/') ?: '/';
}

function buildOpenApiDocument(array $schema, string $basePath): array
{
    $paths  = [];
    $models = [];

    foreach ($schema as $table => $definition) {
        $model = [
            'type'       => 'object',
            'properties' => [],
            'required'   => [],
        ];

        foreach ($definition['columns'] as $column) {
            $model['properties'][$column['Field']] = mapColumnToSchema($column);

            $isNullable = strtoupper($column['Null']) === 'YES';
            $autoIncrement = str_contains(strtolower($column['Extra']), 'auto_increment');

            if (!$isNullable && !$autoIncrement) {
                $model['required'][] = $column['Field'];
            }
        }

        if (!$model['required']) {
            unset($model['required']);
        }

        $models[$table] = $model;

        $paths["/{$table}"] = [
            'get' => [
                'summary'     => "Listar registros de {$table}",
                'description' => 'Devuelve registros con filtros opcionales por columna usando parámetros de consulta.',
                'parameters'  => array_merge(
                    buildColumnParams($definition['columns']),
                    [
                        [
                            'name'        => 'limit',
                            'in'          => 'query',
                            'description' => 'Cantidad máxima de registros (1-500).',
                            'schema'      => ['type' => 'integer', 'default' => 50],
                        ],
                        [
                            'name'        => 'offset',
                            'in'          => 'query',
                            'description' => 'Desplazamiento para paginado.',
                            'schema'      => ['type' => 'integer', 'default' => 0],
                        ],
                    ]
                ),
                'responses'   => [
                    '200' => [
                        'description' => 'Listado exitoso.',
                        'content'     => [
                            'application/json' => [
                                'schema' => [
                                    'type'  => 'object',
                                    'properties' => [
                                        'data' => [
                                            'type'  => 'array',
                                            'items' => ['$ref' => "#/components/schemas/{$table}"],
                                        ],
                                        'meta' => [
                                            'type'       => 'object',
                                            'properties' => [
                                                'table' => ['type' => 'string'],
                                                'filters' => ['type' => 'object', 'additionalProperties' => ['type' => 'string']],
                                            ],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
            'post' => [
                'summary'     => "Crear registro en {$table}",
                'requestBody' => [
                    'required' => true,
                    'content'  => [
                        'application/json' => [
                            'schema' => ['$ref' => "#/components/schemas/{$table}"],
                        ],
                    ],
                ],
                'responses'   => [
                    '201' => [
                        'description' => 'Registro creado.',
                        'content'     => [
                            'application/json' => [
                                'schema' => [
                                    'type'       => 'object',
                                    'properties' => [
                                        'data' => ['$ref' => "#/components/schemas/{$table}"],
                                    ],
                                ],
                            ],
                        ],
                    ],
                ],
            ],
        ];

        if ($definition['primaryKey']) {
            $paramName = $definition['primaryKey'];
            $paths["/{$table}/{" . $paramName . "}"] = [
                'get' => [
                    'summary'     => "Obtener registro de {$table} por {$paramName}",
                    'parameters'  => [
                        [
                            'name'     => $paramName,
                            'in'       => 'path',
                            'required' => true,
                            'schema'   => ['type' => 'string'],
                        ],
                    ],
                    'responses'   => [
                        '200' => [
                            'description' => 'Registro encontrado.',
                            'content'     => [
                                'application/json' => [
                                    'schema' => [
                                        'type'       => 'object',
                                        'properties' => [
                                            'data' => ['$ref' => "#/components/schemas/{$table}"],
                                        ],
                                    ],
                                ],
                            ],
                        ],
                        '404' => [
                            'description' => 'No encontrado.',
                        ],
                    ],
                ],
            ];
        }
    }

    return [
        'openapi' => '3.1.1',
        'info'    => [
            'title'       => 'API Sofi - Instalación automática',
            'description' => 'Especificación generada en base al esquema actual de la base de datos.',
            'version'     => '1.0.0',
        ],
        'servers' => [
            ['url' => $basePath],
        ],
        'paths'      => $paths,
        'components' => ['schemas' => $models],
    ];
}

function buildColumnParams(array $columns): array
{
    $params = [];
    foreach ($columns as $column) {
        $params[] = [
            'name'        => $column['Field'],
            'in'          => 'query',
            'required'    => false,
            'schema'      => mapColumnToSchema($column),
            'description' => 'Filtro exacto por la columna ' . $column['Field'],
        ];
    }

    return $params;
}

function mapColumnToSchema(array $column): array
{
    $type = strtolower($column['Type']);
    $schema = ['type' => 'string'];

    if (preg_match('/int|decimal|float|double|numeric/', $type)) {
        $schema['type'] = str_contains($type, 'int') ? 'integer' : 'number';

        if (str_contains($type, 'big')) {
            $schema['format'] = 'int64';
        } elseif (str_contains($type, 'tinyint(1)')) {
            $schema['type'] = 'boolean';
        } else {
            $schema['format'] = 'int32';
        }
    } elseif (str_contains($type, 'json')) {
        $schema['type'] = 'object';
    } elseif (str_contains($type, 'enum')) {
        $schema['type'] = 'string';
        if (preg_match('/enum\((.+)\)/', $type, $matches)) {
            $options = array_map(static fn($value) => trim($value, " '\""), explode(',', $matches[1]));
            $schema['enum'] = $options;
        }
    } elseif (preg_match('/date|time/', $type)) {
        $schema['type']   = 'string';
        $schema['format'] = str_contains($type, 'time') ? 'time' : 'date-time';
    } elseif (preg_match('/text|char|blob|binary/', $type)) {
        $schema['type'] = 'string';
    }

    if (strtoupper($column['Null']) === 'NO') {
        $schema['nullable'] = false;
    } else {
        $schema['nullable'] = true;
    }

    if ($column['Default'] !== null) {
        $schema['default'] = $column['Default'];
    }

    if (!empty($column['Comment'])) {
        $schema['description'] = $column['Comment'];
    }

    return $schema;
}

function e(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES, 'UTF-8');
}

$prefill = [
    'db_host'    => $_POST['db_host'] ?? 'localhost',
    'db_port'    => $_POST['db_port'] ?? '3306',
    'db_name'    => $_POST['db_name'] ?? '',
    'db_user'    => $_POST['db_user'] ?? '',
    'db_charset' => $_POST['db_charset'] ?? 'utf8mb4',
];
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Instalador Sofi API</title>
    <style>
        body { font-family: Arial, sans-serif; background: #f4f6f8; margin: 0; padding: 40px; }
        .container { max-width: 720px; margin: 0 auto; background: #fff; padding: 32px; border-radius: 12px; box-shadow: 0 15px 35px rgba(0,0,0,0.08); }
        h1 { margin-top: 0; }
        .field { margin-bottom: 16px; }
        label { display: block; font-weight: bold; margin-bottom: 4px; }
        input { width: 100%; padding: 10px; border-radius: 6px; border: 1px solid #cfd6df; }
        button { background: #0057d9; color: #fff; padding: 12px 24px; border: none; border-radius: 6px; cursor: pointer; font-size: 16px; }
        .errors { background: #ffe7e7; color: #831515; padding: 12px; border-radius: 6px; margin-bottom: 16px; }
        .success { background: #e3f7e3; color: #0d6b0d; padding: 12px; border-radius: 6px; margin-bottom: 16px; }
        ul.links { list-style: none; padding-left: 0; }
        ul.links li { margin-bottom: 6px; }
        ul.links a { color: #0057d9; text-decoration: none; }
        .info { font-size: 0.9rem; color: #6c757d; margin-top: 24px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Instalador Sofi API</h1>
        <p>Completa las credenciales MySQL para crear las tablas, configurar la API y generar automáticamente la documentación OpenAPI 3.1.1.</p>

        <?php if ($state['errors']): ?>
            <div class="errors">
                <strong>Revisa los siguientes puntos:</strong>
                <ul>
                    <?php foreach ($state['errors'] as $error): ?>
                        <li><?= e($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <?php if ($state['success']): ?>
            <div class="success">
                <?= e($state['success']['message']) ?>
                <?php if (!empty($state['success']['links'])): ?>
                    <ul class="links">
                        <li><strong>Documentación:</strong> <a href="/docs" target="_blank">Documentacion Swagger</a></li>
                        <?php foreach ($state['success']['links'] as $label => $url): ?>
                            <li><strong><?= e($label) ?>:</strong> <a href="<?= e($url) ?>" target="_blank"><?= e($url) ?></a></li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        <?php endif; ?>

        <form method="post">
            <div class="field">
                <label for="db_host">Host</label>
                <input type="text" id="db_host" name="db_host" value="<?= e($prefill['db_host']) ?>" required>
            </div>
            <div class="field">
                <label for="db_port">Puerto</label>
                <input type="text" id="db_port" name="db_port" value="<?= e($prefill['db_port']) ?>" required>
            </div>
            <div class="field">
                <label for="db_name">Base de datos</label>
                <input type="text" id="db_name" name="db_name" value="<?= e($prefill['db_name']) ?>" required>
            </div>
            <div class="field">
                <label for="db_user">Usuario</label>
                <input type="text" id="db_user" name="db_user" value="<?= e($prefill['db_user']) ?>" required>
            </div>
            <div class="field">
                <label for="db_pass">Contraseña</label>
                <input type="password" id="db_pass" name="db_pass">
            </div>
            <div class="field">
                <label for="db_charset">Charset</label>
                <input type="text" id="db_charset" name="db_charset" value="<?= e($prefill['db_charset']) ?>">
            </div>
            <button type="submit">Instalar</button>
        </form>

        <div class="info">
            El instalador ejecuta <code>mysql_seed.sql</code>, crea el archivo <code>config.php</code>, genera <code>docs/openapi.json</code> y <code>docs/swagger.json</code>, y deja listo el endpoint universal en <code>../index.php</code>.
        </div>
    </div>
</body>
</html>
